// license:BSD-3-Clause
// copyright-holders:m1macrophage

/*
The Sequential Circuits Six-Trak (aka Model 610) is a 6-voice, MIDI-enabled,
digitally-controlled analog synthesizer.

Each voice is built around a CEM3394, a single-oscillator "synth-on-a-chip".
Control voltages (CVs) for each CEM3394 are generated independently, for a
6-part multitimbrality. A single noise source can be mixed in every voice.
Modulation sources (e.g. envelope generators) for the VCA and VCF are computed
by the firmware and incorporated into the CVs. Because of this, the sample &
hold (S&H) circuits for the VCA amplitude and VCF frequency include an
additional RC circuit to smoothen the CV changes.

The synth's computer is built around a Z80. It implements the user interface
(scans for key presses and controls LEDs), implements a sequencer, responds to
and controls MIDI, periodically refreshes CVs for each of the 6 voices, and
takes care of tuning.

CVs are generated by a 12-bit DAC (0V - -4V). A MUX selects between the DAC
output and its inverted output (0V - 4V), resulting in a 13-bit resolution.
However, the full 13-bit resolution is only used for a few CVs. Most CVs just
use 6 bits.

A set of MUXes route the generated CV to the appropriate voice and parameter.
Each of the 6 voice chips (CEM3394) has 8 CV-controlled parameters:
- VCO pitch.
- VCA amplitude.
- VCF resonance.
- VCF cutoff frequency.
- Noise mix.
- VCF cutoff frequency modulation, by the triangle output of the VCO.
- VCO pulse width.
- VCO shape.

Known hardware revisions:
- Model 610 Rev A: serial numbers 1-900.
- Model 610 Rev B: serial numbers 901-?.
  Contrast sixtrak_rev_a() and sixtrak_rev_b() for the differences. In summary:
    * MM5837 noise generator replaced with a transistor-based noise source.
    * Improved autotune circuit.
 - Model 610 Rev C: serial numbers ?-?.
    * No known differences.

This driver is based on the service manual for the Six-Trak (probably Rev C),
and is intended as an educational tool.


*** Tuning and initialization ***

If the NVRAM images are present in the .zip file, the synth will be initialized
to factory settings. The initialization steps below will not be required.

Without default NVRAM content, all programs will be silent, the synth will be
out of tune, and the pitch wheel will not be calibrated. To tune and calibrate,
see below. In this case, patches will need to be programmed manually or
configured via MIDI SYSEX.

A basic test patch can be configured by pressing:
CONTROL RECORD + SELECT 8 (C + NUMPAD 8)

To run the autotune routine, press:
CONTROL RECORD + SELECT 6 (C + NUMPAD 6)

To calibrate the pitch wheel, leave the wheel centered and press:
CONTROL RECORD + SELECT 3 (C + NUMPAD 3)

(Not necessary) Further fine-tuning and tuning-related diagnostics can be done
by entering the "tune test" menu:
CONTROL RECORD + SELECT 9 (C + NUMPAD 9)
More info on "tune test" in the service manual.

To display the firmware version, press:
TRACK RECORD + SELECT 5 (R + NUMPAD 5)

For all other functionality, see the owner's manual.
*/

#include "emu.h"
#include "bus/midi/midi.h"
#include "cpu/z80/z80.h"
#include "machine/6850acia.h"
#include "machine/7474.h"
#include "machine/clock.h"
#include "machine/nvram.h"
#include "machine/output_latch.h"
#include "machine/pit8253.h"
#include "machine/rescap.h"
#include "sound/cem3394.h"
#include "sound/flt_rc.h"
#include "sound/flt_vol.h"
#include "sound/mixer.h"
#include "sound/mm5837.h"
#include "sound/va_eg.h"
#include "speaker.h"

#include "sequential_sixtrak.lh"

#define LOG_CV              (1U << 1)
#define LOG_KEYS            (1U << 2)
#define LOG_ADC_VALUE_KNOB  (1U << 3)
#define LOG_ADC_PITCH_WHEEL (1U << 4)
#define LOG_VOLUME          (1U << 5)
#define LOG_AUTOTUNE        (1U << 6)
#define LOG_CALIBRATION     (1U << 7)
#define LOG_WHEEL_RC        (1U << 8)

#define VERBOSE (LOG_CALIBRATION)
//#define LOG_OUTPUT_FUNC osd_printf_info

#include "logmacro.h"

namespace {

// TODO: Move somewhere in sound/device.
class sixtrak_transistor_noise_device : public device_t, public device_sound_interface
{
public:
	sixtrak_transistor_noise_device(const machine_config &mconfig, const char *tag, device_t *owner, uint32_t clock = 0) ATTR_COLD;

protected:
	void sound_stream_update(sound_stream &stream) override;
	void device_start() override ATTR_COLD;

private:
	sound_stream *m_stream = nullptr;
};

}  // anonymous namespace

DEFINE_DEVICE_TYPE(SIXTRAK_TRANSISTOR_NOISE, sixtrak_transistor_noise_device, "sixtrak_transistor_noise", "Six-Trak PNP-based noise generator")

sixtrak_transistor_noise_device::sixtrak_transistor_noise_device(const machine_config &mconfig, const char *tag, device_t *owner, uint32_t clock)
	: device_t(mconfig, SIXTRAK_TRANSISTOR_NOISE, tag, owner, clock)
	, device_sound_interface(mconfig, *this)
{
}

void sixtrak_transistor_noise_device::sound_stream_update(sound_stream &stream)
{
	const int n = stream.samples();
	for (int i = 0; i < n; ++i)
	{
		// Uniformly distributed random number between -1 and 1.
		stream.put(0, i, 2 * (double(machine().rand()) / std::numeric_limits<u32>::max() - 0.5));
	}
}

void sixtrak_transistor_noise_device::device_start()
{
	m_stream = stream_alloc(0, 1, SAMPLE_RATE_OUTPUT_ADAPTIVE);
}

namespace {

enum wheel_type
{
	WHEEL_PITCH = 0,
	WHEEL_MOD,
};

class sixtrak_state : public driver_device
{
public:
	sixtrak_state(const machine_config &mconfig, device_type type, const char *tag) ATTR_COLD;

	void sixtrak_rev_a(machine_config &config) ATTR_COLD;
	void sixtrak_rev_b(machine_config &config) ATTR_COLD;

	DECLARE_INPUT_CHANGED_MEMBER(wheel_moved);
	DECLARE_INPUT_CHANGED_MEMBER(master_volume_changed);
	DECLARE_INPUT_CHANGED_MEMBER(dac_trimmer_changed);

protected:
	void machine_start() override ATTR_COLD;
	void machine_reset() override ATTR_COLD;

private:
	double get_dac_v(bool inverted) const;
	double get_voltage_mux_out() const;
	static void update_sh_rc(bool sampling, double cv, va_rc_eg_device &rc);
	void update_cvs();

	void update_wheel_rc(int which);
	void update_master_volume();

	void tuning_counter_gate_w(int state);
	void tuning_counter_out_changed(int state);
	void tuning_ff_comp_out_changed(int state);
	void update_tuning_timer();
	TIMER_CALLBACK_MEMBER(tuning_timer_tick);

	void dac_low_w(u8 data);
	void dac_high_w(u8 data);
	void voice_select_w(u8 data);
	void param_select_w(u8 data);
	void digit_w(u8 data);
	u8 misc_r();
	u8 keys_r();

	u64 iorq_wait_state(offs_t offset, u64 current_time);
	void memory_map(address_map &map) ATTR_COLD;
	void io_map(address_map &map) ATTR_COLD;

	void sixtrak_common(machine_config &config, device_sound_interface &noise) ATTR_COLD;

	required_device<z80_device> m_maincpu;
	required_device<pit8253_device> m_pit;
	required_device<ttl7474_device> m_tuning_ff;  // U146A
	required_device<output_latch_device> m_tune_control;  // U148 (CD40174)
	required_device_array<cem3394_device, 6> m_voices;
	required_device_array<va_rc_eg_device, 6> m_gain_rc;
	required_device_array<va_rc_eg_device, 6> m_freq_rc;
	required_device<filter_volume_device> m_mute;  // U147 (CD4049)
	required_device<filter_volume_device> m_master_vol;  // R197
	required_device_array<va_rc_eg_device, 2> m_wheel_rc;
	required_ioport_array<2> m_wheels;
	required_ioport_array<16> m_keys;
	required_ioport m_footswitch;
	required_ioport m_track_vol_knob;
	required_ioport m_speed_knob;
	required_ioport m_value_knob;
	required_ioport m_tune_knob;
	required_ioport m_master_vol_knob;
	required_ioport m_dac_trimmer;
	output_finder<2> m_digits;

	emu_timer *m_tuning_timer = nullptr;
	bool m_tuning_counter_clock_started = false;
	u8 m_tuning_counter_gate = 0;
	u8 m_tuning_counter_out = 0;

	u8 m_key_row = 0;
	u16 m_dac_value = 0;
	u8 m_sh_voices = 0x3f;
	u8 m_sh_param = 0;
	u8 m_voltage_mux_input = 0;
	std::array<bool, 6> m_sampling_gain = { false, false, false, false, false, false };
	std::array<bool, 6> m_sampling_freq = { false, false, false, false, false, false };

	static inline constexpr double VPLUS = 5.0;
	static inline constexpr double VMINUS = -6.5;
};

sixtrak_state::sixtrak_state(const machine_config &mconfig, device_type type, const char *tag)
	: driver_device(mconfig, type, tag)
	, m_maincpu(*this, "maincpu")
	, m_pit(*this, "pit")
	, m_tuning_ff(*this, "tuningff")
	, m_tune_control(*this, "tune_control")
	, m_voices(*this, "cem3394_%u", 1U)
	, m_gain_rc(*this, "gain_rc_%u", 1U)
	, m_freq_rc(*this, "freq_rc_%u", 1U)
	, m_mute(*this, "mute")
	, m_master_vol(*this, "master_volume")
	, m_wheel_rc(*this, "wheel_rc_%u", 0U)
	, m_wheels(*this, "wheel_%u", 0U)
	, m_keys(*this, "key_row_%u", 0U)
	, m_footswitch(*this, "footswitch")
	, m_track_vol_knob(*this, "track_volume_knob")
	, m_speed_knob(*this, "speed_knob")
	, m_value_knob(*this, "value_knob")
	, m_tune_knob(*this, "tune_knob")
	, m_master_vol_knob(*this, "master_volume_knob")
	, m_dac_trimmer(*this, "trimmer_dac_inverter")
	, m_digits(*this, "digit_%u", 1U)
{
}

double sixtrak_state::get_dac_v(bool inverted) const
{
	// The 12-bit DAC (U110, 7541) and corresponding I2V converter (U156, LF411A)
	// produce a voltage in the range 0V (all data bits 0) to -4V (all data
	// bits 1). That voltage is inverted by U111A (5532) and surrounding
	// resistors.

	// A MUX (U108, 4051) selects between the non-inverted and inverted voltages
	// when updating CVs. This essentially adds 1 bit of precision to the
	// 12-bit DAC, for the CVs that need it. But most CVs just use 6 bits. The
	// ADC comparator always compares against the inverted (0V - 4V) voltage.

	constexpr double DAC_MAX_VOLTAGE = -4.0;
	constexpr double DAC_MAX_VALUE = double(make_bitmask<u16>(12));
	const double v = m_dac_value * DAC_MAX_VOLTAGE / DAC_MAX_VALUE;

	if (inverted)
	{
		// Inversion done by U111A (5532 op-amp) and surrounding resistors.
		constexpr double R121 = RES_K(3.01);
		constexpr double R123 = RES_K(3.01);
		constexpr double R1205 = RES_K(100);  // trimmer
		constexpr double R1206 = RES_M(2.2);

		const double rp1 = m_dac_trimmer->read() * R1205 / m_dac_trimmer->field(1)->maxval();
		const double rp2 = R1205 - rp1;
		// Compute voltage at the junction of all resistors.
		const double vx = (R1206 * rp1 * VMINUS + R1206 * rp2 * VPLUS) / (R1206 * rp1 + R1206 * rp2 + rp1 * rp2);
		return -R123 * (v / R121 + vx / R1206);  // Voltage at U111A's output.
	}
	else
	{
		return v;
	}
}

double sixtrak_state::get_voltage_mux_out() const
{
	// All knobs are 10K linear potentiometers, and each one is attached to
	// ground on one side and to a separate 2K resistor to 5V on the other. The
	// pot wipers are attached to corresponding mux inputs.
	constexpr double KNOB_MAX_V = VPLUS * RES_VOLTAGE_DIVIDER(RES_K(2), RES_K(10));

	// The voltage MUX (U108, CD4051) routes potentiometer voltages to the ADC
	// comparator and routes the appropriate DAC output to the CV S&H circuits.
	// The MUX's output is always enabled. The firmware controls whether the ADC
	// or an S&H circuit is activated.
	switch (m_voltage_mux_input)
	{
		case 0: return get_dac_v(false);
		case 1: return get_dac_v(true);
		case 2: return m_value_knob->read() * KNOB_MAX_V / m_value_knob->field(1)->maxval();
		case 3: return m_tune_knob->read() * KNOB_MAX_V / m_tune_knob->field(1)->maxval();
		case 4: return m_wheel_rc[WHEEL_MOD]->get_v();
		case 5: return m_track_vol_knob->read() * KNOB_MAX_V / m_track_vol_knob->field(1)->maxval();
		case 6: return m_wheel_rc[WHEEL_PITCH]->get_v();
		case 7: return m_speed_knob->read() * KNOB_MAX_V / m_speed_knob->field(1)->maxval();
	}

	assert(false);  // Execution should not reach here.
	return 0;
}

// Will only be accurate if called when `sampling` is true, or when `sampling`
// is transitioning to false. Should not be called multiple times in a row with
// sampling == false.
void sixtrak_state::update_sh_rc(bool sampling, double cv, va_rc_eg_device &rc)
{
	// The sample & hold circuits for the filter frequency and VCA gain CVs
	// include an RC network. This smoothens the CV updates sent from the
	// firmware.

	// When the CV is being sampled, Cl (Clarge) reaches the target voltage
	// immediately, while Cs (Csmall) (dis)charges towards the CV via R (1 MOhm).
	// The CEM3394 senses the voltage at Cs.
	//
	// CV ---+--- R ---+--- CEM3394 CV input
	//       |         |
	//       Cl        Cs
	//       |         |
	//      GND       GND

	// When the CV is not being sampled, Cs and Cl will (dis)charge towards the
	// same target voltage. The (dis)charge rate will be that of an RC circuit
	// where C is the series combination of Cs and Cl.
	//
	//       +--- R ---+--- CEM3394 CV input
	//       |         |
	//       Cl        Cs
	//       |         |
	//      GND       GND

	constexpr double C_SMALL = CAP_U(0.001);
	constexpr double C_LARGE = CAP_U(0.01);
	constexpr double C_SERIES = (C_LARGE * C_SMALL) / (C_LARGE + C_SMALL);

	if (sampling)
	{
		rc.set_target_v(cv);
		rc.set_c(C_SMALL);
	}
	else
	{
		rc.set_target_v((C_LARGE * rc.get_target_v() + C_SMALL * rc.get_v()) / (C_LARGE + C_SMALL));
		rc.set_c(C_SERIES);
	}
}

void sixtrak_state::update_cvs()
{
	constexpr int PARAM2CVIN[8] =
	{
		cem3394_device::VCO_FREQUENCY,
		cem3394_device::FINAL_GAIN,
		cem3394_device::FILTER_RESONANCE,
		cem3394_device::FILTER_FREQUENCY,
		cem3394_device::MIXER_BALANCE,
		cem3394_device::MODULATION_AMOUNT,
		cem3394_device::PULSE_WIDTH,
		cem3394_device::WAVE_SELECT,
	};

	constexpr const char *PARAMNAMES[8] =
	{
		"pitch", "gain", "resonance", "cutoff", "mixer", "mod", "PW", "waveform"
	};

	assert(m_sh_param < 8);
	const int cv_input = PARAM2CVIN[m_sh_param];
	const double cv = get_voltage_mux_out();

	for (int voice = 0; voice < 6; ++voice)
	{
		const bool voice_active = !BIT(m_sh_voices, voice);

		const bool sampling_gain = voice_active && (m_sh_param == 1);
		if (sampling_gain || sampling_gain != m_sampling_gain[voice])
		{
			update_sh_rc(sampling_gain, cv, *m_gain_rc[voice]);
		}
		m_sampling_gain[voice] = sampling_gain;

		const bool sampling_freq = voice_active && (m_sh_param == 3);
		if (sampling_freq || sampling_freq != m_sampling_freq[voice])
		{
			update_sh_rc(sampling_freq, cv, *m_freq_rc[voice]);
		}
		m_sampling_freq[voice] = sampling_freq;

		if (!voice_active || sampling_freq || sampling_gain)
			continue;

		cem3394_device *v = m_voices[voice];
		if (v->get_voltage(cv_input) == cv)
			continue;

		v->set_voltage(cv_input, cv);
		LOGMASKED(LOG_CV, "CV - voice: %u, param: %u (%s), cv: %f - %03x - %x\n",
				  voice, m_sh_param, PARAMNAMES[m_sh_param], cv, m_dac_value, m_voltage_mux_input);
		if (m_sh_param == 0)
			LOGMASKED(LOG_CV, "Pitch %d: %f\n", voice, v->get_parameter(cem3394_device::VCO_FREQUENCY));
	}
}

void sixtrak_state::update_wheel_rc(int which)
{
	// The pitch and mod wheel potentiometers (100K, linear) are attached to
	// ground on one side and a shared 27K resistor (R1208) to 5V on the other.
	// The pot wipers are attached to corresponding mux inputs and capacitors.

	// The diagram on the left is for the pitch wheel (R1) RC circuit. The mod
	// wheel circuit is the same, but the capacitor is connected to R2 and a
	// different MUX input. The circuit to the right is the equivalent RC
	// circuit. For the derivation of Veq and Req, see equations below.

	//     V+                                        Veq
	//     |                                          |
	//   R1208                                        |
	//     |                                         Req
	//     +------+                                   |
	//     |      |                                   |
	//     |     R1t     A                            +----- MUX input
	//     R2     +------+------ MUX input            |
	//     |     R1b     C                            C
	//     |      |      |                            |
	//    GND    GND    GND                          GND

	assert(which == WHEEL_PITCH || which == WHEEL_MOD);
	constexpr const char *WHEEL_NAME[2] = {"Pitch", "Mod"};

	// The real hardware does not use the full range of the wheel potentiometers.
	// Using the full range results in erratic behavior. The exact range is not
	// known, but the value used here results in reasonable pitch bend behavior
	// (+/- ~3 semitones).
	constexpr double WHEEL_RANGE = 0.25;

	const s32 value = m_wheels[which]->read();
	if (value <= 0)
	{
		m_wheel_rc[which]->set_instant_v(0);
		LOGMASKED(LOG_WHEEL_RC, "%s wheel RC - V: 0\n", WHEEL_NAME[which]);
		return;
	}

	// The equations below are for the pitch wheel (R1). But they also work for
	// the mod wheel (R2) because R1 == R2.

	constexpr double R1 = RES_K(100);
	constexpr double R2 = RES_K(100);
	constexpr double R1208 = RES_K(27);

	const double r1_bottom = value * WHEEL_RANGE * R1 / m_wheels[which]->field(1)->maxval();
	const double r1_top = R1 - r1_bottom;

	// To compute Veq, calculate the voltage at point A, ignoring the connection
	// to the capacitor.
	const double v_eq = VPLUS * RES_VOLTAGE_DIVIDER(R1208, RES_2_PARALLEL(R1, R2)) * RES_VOLTAGE_DIVIDER(r1_top, r1_bottom);

	// To compute Req, consider V+ and GND as being connected to each other, and
	// calculate the resistance to that node. Ignore the connection to the
	// capacitor.
	const double r_eq = RES_2_PARALLEL(r1_bottom, r1_top + RES_2_PARALLEL(R1208, R2));

	m_wheel_rc[which]->set_r(r_eq);
	m_wheel_rc[which]->set_target_v(v_eq);

	LOGMASKED(LOG_WHEEL_RC, "%s wheel RC - R: %f, V: %f\n", WHEEL_NAME[which], r_eq, v_eq);
}

void sixtrak_state::update_master_volume()
{
	const double proportion = double(m_master_vol_knob->read()) / m_master_vol_knob->field(1)->maxval();
	const double gain = RES_AUDIO_POT_LAW(proportion);
	m_master_vol->set_gain(gain);
	LOGMASKED(LOG_VOLUME, "Set volume to: %f %f\n", proportion, gain);
}

void sixtrak_state::tuning_counter_gate_w(int state)
{
	m_tuning_counter_gate = state;
	if (!m_tuning_counter_gate)
		m_tuning_counter_clock_started = false;

	m_pit->write_gate0(m_tuning_counter_gate);

	if (m_tuning_counter_gate)
	{
		LOGMASKED(LOG_AUTOTUNE, "Autotune routine started.\n");
		update_tuning_timer();
	}
}

void sixtrak_state::tuning_counter_out_changed(int state)
{
	m_tuning_counter_out = state;
	m_pit->write_gate1(m_tuning_counter_out ? 0 : 1);  // Inverted by U151D (74LS04).

	if (m_tuning_counter_out)
	{
		m_tuning_timer->reset();
		LOGMASKED(LOG_AUTOTUNE, "Autotune routine ended.\n");
	}
}

void sixtrak_state::tuning_ff_comp_out_changed(int state)
{
	// The autotune implementation on the Six-Trak depends on behavior of the
	// 8253 that does not seem to be accurately emulated in pit8253.cpp.
	// Specifically, the Six-Trak expects that:
	// - The counter is incremented on the negative-going edge of the clock.
	// - In mode 1, counting will start after the first *full* clock cycle
	//   (positive-going edge followed by negative-going) after the gate is
	//   enabled.

	// This implementation works around those issues as follows:
	// - The signal to the CLK0 PIT input is inverted.
	// - Using `m_tuning_counter_clock_started` to detect the first positive-
	//   going clock edge after the gate is enabled, and only clocking the PIT
	//   after this has occurred.

	if (m_tuning_counter_gate && state)
		m_tuning_counter_clock_started = true;

	if (m_tuning_counter_clock_started)
		m_pit->write_clk0(!state);
}

void sixtrak_state::update_tuning_timer()
{
	// The tuning circuit uses a comparator (U144, LM311) that senses the
	// mixed voice signal and clocks the tuning flipflop (U146A, 74LS174) at the
	// frequency of that signal. The flipflop and a PIT (U133A + U133B, 8253)
	// are configured to measure the time it takes for a predetermined number
	// of signal cycles to occur.

	// Tuning is performed by enabling one voice at a time and measuring the
	// frequencies of the VCO and filter. To measure the frequency of the
	// filter, the firmware sets the resonance to a high value, to cause
	// self-oscillation at the cutoff frequency.

	// This implementation approximates the above by using a timer to clock the
	// flipflop at the frequency of the oscillator or filter. The timer
	// frequency is determined by finding the loudest voice, and either using
	// the frequency of the VCO, or the filter, depending on whether resonance
	// is set to a high value.

	int active_voices = 0;
	int loudest_voice = -1;
	double max_gain_cv = -1;
	for (int voice = 0; voice < m_voices.size(); ++voice)
	{
		const double gain_cv = m_voices[voice]->get_voltage(cem3394_device::FINAL_GAIN);
		if (gain_cv > 0.1)
			++active_voices;
		if (gain_cv > max_gain_cv)
		{
			max_gain_cv = gain_cv;
			loudest_voice = voice;
		}
	}

	if (loudest_voice < 0)
	{
		m_tuning_timer->reset();
		logerror("Autotune: no voice selected.\n");
		return;
	}
	else if (active_voices > 1)
	{
		logerror("Autotune: multiple voices selected. Using the loudest one.\n");
	}

	cem3394_device *voice = m_voices[loudest_voice];
	double freq = 0;
	bool tuning_filter = false;
	if (voice->get_parameter(cem3394_device::FILTER_RESONANCE) > 0.9)
	{
		freq = voice->get_parameter(cem3394_device::FILTER_FREQUENCY);
		tuning_filter = true;
	}
	else
	{
		freq = voice->get_parameter(cem3394_device::VCO_FREQUENCY);
		tuning_filter = false;
	}

	const attotime t = attotime::from_hz(freq);
	m_tuning_timer->adjust(t, 0, t);
	LOGMASKED(LOG_AUTOTUNE, "Autotuning voice: %d is_filter: %d frequency: %f\n",
			  loudest_voice, tuning_filter, freq);
}

TIMER_CALLBACK_MEMBER(sixtrak_state::tuning_timer_tick)
{
	m_tuning_ff->clock_w(1);
	m_tuning_ff->clock_w(0);
}

void sixtrak_state::dac_low_w(u8 data)
{
	// D2-D7 -> Latch U105 -> low 6 bits of 12-bit DAC (U110, 7541).
	m_dac_value = (m_dac_value & 0x0fc0) | ((data >> 2) & 0x3f);
	update_cvs();
}

void sixtrak_state::dac_high_w(u8 data)
{
	// D0-D5 -> Latch U104 -> high 6 bits of 12-bit DAC (U110, 7541).
	m_dac_value = (u16(data & 0x3f) << 6) | (m_dac_value & 0x003f);
	update_cvs();
}

void sixtrak_state::voice_select_w(u8 data)
{
	// D0-D5 -> Latch U120 -> INH input of 6 x 4051.
	// Selects which of the voice MUXes to activate.
	m_sh_voices = data & 0x3f;
	update_cvs();
}

void sixtrak_state::param_select_w(u8 data)
{
	// D0-D2 -> Latch U118 D0-D2 -> A, B, C inputs of all 6 S&H 4051.
	// Selects which voice parameter to activate.
	m_sh_param = data & 0x07;

	// D7-D5 -> Latch U118 D3-D5 -> A, B, C inputs of Voltage MUX 4051 (U110).
	m_voltage_mux_input = bitswap<3>(data, 5, 6, 7);

	// D4-D7 -> Decoder U155 (CD4514) -> key row select.
	m_key_row = (data >> 4) & 0x0f;

	update_cvs();
}

void sixtrak_state::digit_w(u8 data)
{
	static constexpr u8 PATTERNS_CD4511[0x10] =
	{
		0x3f, 0x06, 0x5b, 0x4f, 0x66, 0x6d, 0x7c, 0x07, 0x7f, 0x67, 0, 0, 0, 0, 0, 0
	};
	m_digits[0] = PATTERNS_CD4511[BIT(data, 0, 4)];  // left
	m_digits[1] = PATTERNS_CD4511[BIT(data, 4, 4)];  // right
}

u8 sixtrak_state::misc_r()
{
	// D0, D1 - autotune-related.
	const u8 d0 = m_tuning_ff->output_comp_r();
	const u8 d1 = m_tuning_counter_out;

	// D2: ADC comparator.
	// If either of the MUX inputs B or C are high, they will activate the
	// ADC comparator's (U109, LM311) output pullup resistor (R168) via diodes
	// D128 and D129.
	const bool comp_enabled = BIT(m_voltage_mux_input, 1, 2);
	const u8 d2 = (comp_enabled && get_voltage_mux_out() >= get_dac_v(true)) ? 1 : 0;
	if (comp_enabled)
	{
		if (m_voltage_mux_input == 2)
		{
			LOGMASKED(LOG_ADC_VALUE_KNOB, "ADC value - input: %d, pot v: %f, dac v: %f, comp: %d\n",
					  m_value_knob->read(), get_voltage_mux_out(), get_dac_v(true), d2);
		}
		else if (m_voltage_mux_input == 6)
		{
			LOGMASKED(LOG_ADC_PITCH_WHEEL, "ADC pitch - input: %d, pot v: %f, dac v: %f, comp: %d\n",
					  m_wheels[WHEEL_PITCH]->read(), get_voltage_mux_out(), get_dac_v(true), d2);
		}
	}

	// D3: Control footswitch.
	const u8 d3 = BIT(m_footswitch->read(), 0);

	return (d3 << 3) | (d2 << 2) | (d1 << 1) | d0;
}

u8 sixtrak_state::keys_r()
{
	u8 data = m_keys[m_key_row]->read();
	if (m_key_row < 4)
	{
		// The first 4 rows of the key matrix do not have protection diodes for
		// each key. If a key is pressed in the row being scanned, and one or
		// more keys in the same column (within rows 0-3) are pressed, there will
		// be a short between a high and low output of the CD4514 (assuming
		// the schematic is not missing diodes on those outputs), and the press
		// will likely not register. The first 4 rows are used for the synth's
		// buttons. The piano roll keys have the typical diodes and don't have
		// this issue. Furthermore, there are diodes protecting the rest of the
		// rows from rows 0-3.
		u8 inactive_row_presses = 0;
		for (int row = 0; row < 4; ++row)
			if (row != m_key_row)
				inactive_row_presses |= m_keys[row]->read();
		if (data & inactive_row_presses)
			LOGMASKED(LOG_KEYS, "Conflicting key presses: %02x %02x\n", data, inactive_row_presses);
		data = data & ~inactive_row_presses;
	}
	if (data)
		LOGMASKED(LOG_KEYS, "Key row: %d, value: %02x\n", m_key_row, data);
	return data;
}

u64 sixtrak_state::iorq_wait_state(offs_t offset, u64 current_time)
{
	// U123C and U123D (74LS02) will set /WAIT low when both /IORQ and the 2MHz
	// clock are low. So the 2MHz clock needs to be high for an IORQ to proceed.
	// The 2MHz clock is the CPU's clock (4MHz) divided by 2.
	return current_time | 1;
}

void sixtrak_state::memory_map(address_map &map)
{
	map(0x0000, 0x3fff).rom();  // U128 (27128)
	map(0x4000, 0x4fff).ram().share("nvram_a");  // U119, U117 (6116)
	map(0x5000, 0x57ff).mirror(0x0800).ram().share("nvram_b");  // U112 (6116)
	map(0x6000, 0x6001).mirror(0x1ffe).w("midiacia", FUNC(acia6850_device::write));
	map(0xe000, 0xe001).mirror(0x1ffe).r("midiacia", FUNC(acia6850_device::read));
}

void sixtrak_state::io_map(address_map &map)
{
	map.global_mask(0xff);

	map(0x00, 0x03).mirror(0xf4).rw(m_pit, FUNC(pit8253_device::read), FUNC(pit8253_device::write));

	map(0x09, 0x09).mirror(0xf6).r(FUNC(sixtrak_state::misc_r));
	map(0x0a, 0x0a).mirror(0xf5).r(FUNC(sixtrak_state::keys_r));

	map(0x08, 0x08).mirror(0xe0).w("led_latch_2", FUNC(output_latch_device::write));
	map(0x09, 0x09).mirror(0xf0).w(FUNC(sixtrak_state::dac_low_w));
	map(0x0a, 0x0a).mirror(0xf0).w(FUNC(sixtrak_state::dac_high_w));
	map(0x0b, 0x0b).mirror(0xf0).w("led_latch_0", FUNC(output_latch_device::write));
	map(0x0c, 0x0c).mirror(0xf0).w("led_latch_1", FUNC(output_latch_device::write));
	map(0x0d, 0x0d).mirror(0xf0).w(m_tune_control, FUNC(output_latch_device::write));
	map(0x0e, 0x0e).mirror(0xf0).w(FUNC(sixtrak_state::voice_select_w));
	map(0x0f, 0x0f).mirror(0xf0).w(FUNC(sixtrak_state::param_select_w));
	map(0x18, 0x18).mirror(0xe0).w(FUNC(sixtrak_state::digit_w));
}

void sixtrak_state::machine_start()
{
	save_item(NAME(m_tuning_counter_clock_started));
	save_item(NAME(m_tuning_counter_gate));
	save_item(NAME(m_tuning_counter_out));
	save_item(NAME(m_key_row));
	save_item(NAME(m_dac_value));
	save_item(NAME(m_sh_voices));
	save_item(NAME(m_sh_param));
	save_item(NAME(m_voltage_mux_input));
	save_item(NAME(m_sampling_gain));
	save_item(NAME(m_sampling_freq));

	m_digits.resolve();

	m_maincpu->space(AS_IO).install_readwrite_before_time(
		0x00, 0xff, ws_time_delegate(*this, FUNC(sixtrak_state::iorq_wait_state)));

	m_tuning_timer = timer_alloc(FUNC(sixtrak_state::tuning_timer_tick), this);
}

void sixtrak_state::machine_reset()
{
	update_wheel_rc(WHEEL_PITCH);
	update_wheel_rc(WHEEL_MOD);
	update_master_volume();
}

void sixtrak_state::sixtrak_common(machine_config &config, device_sound_interface &noise)
{
	Z80(config, m_maincpu, 8_MHz_XTAL / 2);  // U134 (74LS93), QA.
	m_maincpu->set_addrmap(AS_PROGRAM, &sixtrak_state::memory_map);
	m_maincpu->set_addrmap(AS_IO, &sixtrak_state::io_map);

	NVRAM(config, "nvram_a", nvram_device::DEFAULT_ALL_0);  // U119, U117 (6116)
	NVRAM(config, "nvram_b", nvram_device::DEFAULT_ALL_0);  // U112 (6116)

	PIT8253(config, m_pit);  // U133
	m_pit->set_clk<1>(8_MHz_XTAL / 4);  // U134 (74LS93), QB.
	m_pit->set_clk<2>(8_MHz_XTAL / 4);  // U134 (74LS93), QB.
	m_pit->out_handler<0>().set(FUNC(sixtrak_state::tuning_counter_out_changed));
	m_pit->out_handler<2>().set_inputline(m_maincpu, INPUT_LINE_IRQ0);

	auto &aciaclock = CLOCK(config, "aciaclock", 8_MHz_XTAL / 16);  // U134 (74LS93), QD.
	aciaclock.signal_handler().set("midiacia", FUNC(acia6850_device::write_txc));
	aciaclock.signal_handler().append("midiacia", FUNC(acia6850_device::write_rxc));
	aciaclock.signal_handler().append("nmiff", FUNC(ttl7474_device::clock_w));

	auto &acia = ACIA6850(config, "midiacia", 0);  // U137 (or is it U157?)
	acia.txd_handler().set("mdout", FUNC(midi_port_device::write_txd));
	acia.irq_handler().set("nmiff", FUNC(ttl7474_device::d_w)).invert();
	acia.write_dcd(0);
	acia.write_cts(0);

	TTL7474(config, "nmiff", 0).output_cb().set_inputline(m_maincpu, INPUT_LINE_NMI).invert();  // U146B

	TTL7474(config, m_tuning_ff, 0);
	m_tuning_ff->comp_output_cb().set(FUNC(sixtrak_state::tuning_ff_comp_out_changed));

	MIDI_PORT(config, "mdin", midiin_slot, "midiin").rxd_handler().set("midiacia", FUNC(acia6850_device::write_rxd));
	MIDI_PORT(config, "mdout", midiout_slot, "midiout");

	config.set_default_layout(layout_sequential_sixtrak);

	OUTPUT_LATCH(config, m_tune_control);  // U148, CD40174
	m_tune_control->bit_handler<0>().set([this] (int state) { m_mute->set_gain(state ? 1.0 : 0.0); });
	m_tune_control->bit_handler<1>().set(FUNC(sixtrak_state::tuning_counter_gate_w));
	m_tune_control->bit_handler<2>().set(m_tuning_ff, FUNC(ttl7474_device::preset_w));
	m_tune_control->bit_handler<4>().set(m_tuning_ff, FUNC(ttl7474_device::clear_w));
	m_tune_control->bit_handler<5>().set("nmiff", FUNC(ttl7474_device::preset_w));  // NMI disable.

	auto &u102 = OUTPUT_LATCH(config, "led_latch_0");  // 74LS174
	u102.bit_handler<0>().set_output("led_track_1").invert();
	u102.bit_handler<1>().set_output("led_track_2").invert();
	u102.bit_handler<2>().set_output("led_track_3").invert();
	u102.bit_handler<3>().set_output("led_track_4").invert();
	u102.bit_handler<4>().set_output("led_track_5").invert();
	u102.bit_handler<5>().set_output("led_track_6").invert();

	auto &u101 = OUTPUT_LATCH(config, "led_latch_1");  // 74LS174
	u101.bit_handler<0>().set_output("led_bottom_1").invert();  // led_seq_a
	u101.bit_handler<1>().set_output("led_bottom_2").invert();  // led_sq_b
	u101.bit_handler<2>().set_output("led_bottom_3").invert();  // led_arp_up_down
	u101.bit_handler<3>().set_output("led_bottom_4").invert();  // led_arp_assign
	u101.bit_handler<4>().set_output("led_bottom_5").invert();  // led_stack_a
	u101.bit_handler<5>().set_output("led_bottom_6").invert();  // led_stack_b

	auto &u149 = OUTPUT_LATCH(config, "led_latch_2");  // 74LS174
	u149.bit_handler<0>().set_output("led_control_1").invert();  // led_program
	u149.bit_handler<1>().set_output("led_control_2").invert();  // led_param
	u149.bit_handler<2>().set_output("led_control_3").invert();  // led_value
	u149.bit_handler<3>().set_output("led_record").invert();
	u149.bit_handler<4>().set_output("led_record_track").invert();
	u149.bit_handler<5>().set_output("led_legato").invert();

	// While the capacitor for the pitch wheel is large enough to make a
	// difference (it smoothens pitch bends), the one for the mod is small and
	// possibly there just to suppress noise.
	VA_RC_EG(config, m_wheel_rc[WHEEL_PITCH]).set_c(CAP_U(2.2));  // C110
	VA_RC_EG(config, m_wheel_rc[WHEEL_MOD]).set_c(CAP_U(0.1));  // C111


	// *** Audio configuration ***

	// The audio pipeline operates on current and voltage magnitudes. This
	// scaler converts the loudest voltage signal to an audio signal within the
	// range [-1, 1].
	constexpr double VOLTAGE_TO_AUDIO_SCALER = 0.7;

	// The typical peak-to-peak current of the CE33394, when all waveforms are
	// enabled, is 400 uA [-200, 200].
	constexpr double CEM3394_IOUT_MAX = 200E-6;

	// Using some jitter for the VCO capacitor values, for realism. Otherwise,
	// unison just sounds like a louder oscillator. Using precomputed random
	// values to ensure determinism, and because changing these requires retuning.
	// Random values generated in python: [random.uniform(-1, 1) for i in range(0, 6)]
	constexpr double C_VCO_JITTER[6] = {-0.9781, 0.0426, 0.6231, -0.4256, 0.2138, -0.0607};
	constexpr double C_VCO = CAP_U(0.002);

	for (int i = 0; i < 6; ++i)
	{
		noise.add_route(0, m_voices[i], 1.0, cem3394_device::AUDIO_INPUT);

		VA_RC_EG(config, m_gain_rc[i]).set_r(RES_M(1));
		m_gain_rc[i]->add_route(0, m_voices[i], 1.0, cem3394_device::FINAL_GAIN);

		VA_RC_EG(config, m_freq_rc[i]).set_r(RES_M(1));
		m_freq_rc[i]->add_route(0, m_voices[i], 1.0, cem3394_device::FILTER_FREQUENCY);

		CEM3394(config, m_voices[i]);
		const double c_vco = C_VCO + C_VCO * C_VCO_JITTER[i] * 0.025;  // +/- 2.5%.
		m_voices[i]->configure(RES_K(301), c_vco, CAP_U(0.033), CAP_U(10));
		m_voices[i]->add_route(0, "voicemixer", CEM3394_IOUT_MAX);
	}

	// The output currents of all CEM3394 chips are summed and converted to a
	// voltage by U145B (TL082) and R133.
	// The output of the mixer is connected to:
	// * Comparator U144 (LM311). Its function is emulated with a timer. See
	//   update_tuning_timer() for details.
	// * A muting circuit. See below.
	MIXER(config, "voicemixer").add_route(0, m_mute, -RES_K(4.7));  // R133

	// A circuit built around a CMOS inverter (U147, CD4049), used in an
	// unconventional way to achieve noiseless muting. The Six-Trak service
	// manual explains this in detail. In summary: when sound is enabled,
	// current from R162 will flow (through U147) to the summing node of U145A
	// (TL082), and converted back to a voltage via R165. When disabled, that
	// current will be mostly blocked, and the gain of U145A will be reduced to
	// silence any current that makes it through.
	constexpr double R162 = RES_K(100);
	constexpr double R165 = RES_K(100);
	FILTER_VOLUME(config, m_mute).add_route(0, m_master_vol, (1.0 / R162) * -R165);

	// Master volume is controlled by R197, a 10K audio-taper knob on the front panel.
	FILTER_VOLUME(config, m_master_vol).add_route(0, "mono", VOLTAGE_TO_AUDIO_SCALER);

	SPEAKER(config, "mono").front_center();
}

void sixtrak_state::sixtrak_rev_a(machine_config &config)
{
	// In contrast to the Rev B, the Rev A uses an MM5837 as the noise source.
	// Could not find a schematic for Rev A, so using the configuration and
	// component values from the Sente voice board, which is based on the
	// Six-Trak.

	auto &noise = MM5837_STREAM(config, "noise");
	// The actual VDD is -6.5. But according to comments on sente6vb.cpp, that
	// sounds too low, and it is possible the mapping in mm5837 is wrong. Using
	// -8.0 as per sente6vb.cpp.
	noise.set_vdd(-8.0);
	noise.add_route(ALL_OUTPUTS, "ac_noise", 1.0);

	auto &ac_noise = FILTER_RC(config, "ac_noise");
	ac_noise.set_rc(filter_rc_device::HIGHPASS, RES_K(68) + RES_K(1), 0, 0, CAP_U(2.2));

	sixtrak_common(config, ac_noise);

	// On the Rev A, the D input of the tuning flip-flop is controlled by the
	// firmware.
	m_tune_control->bit_handler<3>().set([this] (int state) { m_tuning_ff->d_w(state); });
}

void sixtrak_state::sixtrak_rev_b(machine_config &config)
{
	sixtrak_common(config, SIXTRAK_TRANSISTOR_NOISE(config, "noise"));

	// On the Rev B and C, the D input of the tuning flip-flop is attached to
	// its /Q output.
	m_tuning_ff->comp_output_cb().append([this] (int state) { m_tuning_ff->d_w(state); });
}

DECLARE_INPUT_CHANGED_MEMBER(sixtrak_state::wheel_moved)
{
	update_wheel_rc(param);
}

DECLARE_INPUT_CHANGED_MEMBER(sixtrak_state::master_volume_changed)
{
	update_master_volume();
}

DECLARE_INPUT_CHANGED_MEMBER(sixtrak_state::dac_trimmer_changed)
{
	update_cvs();

	const u16 prev_dac_value = m_dac_value;
	m_dac_value = 0;
	LOGMASKED(LOG_CALIBRATION, "DAC inverter trimmer adjusted ( 0V). V: %f, -V: %f, offset: %f\n",
			  get_dac_v(false), get_dac_v(true), get_dac_v(false) - get_dac_v(true));
	m_dac_value = 0x0fff;
	LOGMASKED(LOG_CALIBRATION, "DAC inverter trimmer adjusted (-4V). V: %f, -V: %f\n",
			  get_dac_v(false), get_dac_v(true));
	m_dac_value = prev_dac_value;
}

INPUT_PORTS_START(sixtrak)
	PORT_START("key_row_0")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("SEL 0") PORT_CODE(KEYCODE_0_PAD)
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("SEL 1") PORT_CODE(KEYCODE_1_PAD)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("SEL 2") PORT_CODE(KEYCODE_2_PAD)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("SEL 3") PORT_CODE(KEYCODE_3_PAD)
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("SEL 4") PORT_CODE(KEYCODE_4_PAD)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("SEL 5") PORT_CODE(KEYCODE_5_PAD)
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("SEL 6") PORT_CODE(KEYCODE_6_PAD)
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("SEL 7") PORT_CODE(KEYCODE_7_PAD)

	PORT_START("key_row_1")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("SEL 8") PORT_CODE(KEYCODE_8_PAD)
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("SEL 9") PORT_CODE(KEYCODE_9_PAD)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("PROGRAM") PORT_CODE(KEYCODE_G)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("PARAM") PORT_CODE(KEYCODE_P)
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("VALUE") PORT_CODE(KEYCODE_V)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("RECORD") PORT_CODE(KEYCODE_C)
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("RECORD TRACK") PORT_CODE(KEYCODE_R)
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("LEGATO")

	PORT_START("key_row_2")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("TRACK 1") PORT_CODE(KEYCODE_1)
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("TRACK 2") PORT_CODE(KEYCODE_2)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("TRACK 3") PORT_CODE(KEYCODE_3)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("TRACK 4") PORT_CODE(KEYCODE_4)
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("TRACK 5") PORT_CODE(KEYCODE_5)
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("TRACK 6") PORT_CODE(KEYCODE_6)
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("key_row_3")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("SEQ A")
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("SEQ B")
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("UP/DOWN")
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("ARP ASSIGN")
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("STACK A")
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_KEYPAD) PORT_NAME("STACK B")
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("key_row_4")
	PORT_BIT(0xff, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("key_row_5")
	PORT_BIT(0xff, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("key_row_6")
	PORT_BIT(0xff, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("key_row_7")
	PORT_BIT(0xff, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("key_row_8")  // C0 - G0 in schematic.
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_C2
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_CS2
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_D2
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_DS2
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_E2
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_F2
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_FS2
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_G2

	PORT_START("key_row_9")  // G#0 - D#1
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_GS2
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_A2
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_AS2
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_B2
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_C3
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_CS3
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_D3
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_DS3

	PORT_START("key_row_10")  // E1 - B1
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_E3
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_F3
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_FS3
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_G3
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_GS3
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_A3
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_AS3
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_B3

	PORT_START("key_row_11")  // C2 - G2
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_C4
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_CS4
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_D4
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_DS4
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_E4
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_F4
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_FS4
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_G4

	PORT_START("key_row_12")  // G#2 - D#3
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_GS4
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_A4 PORT_CODE(KEYCODE_Z)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_AS4
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_B4
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_C5
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_CS5
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_D5
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_DS5

	PORT_START("key_row_13")  // E3 - B3
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_E5
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_F5
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_FS5
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_G5
	PORT_BIT(0x10, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_GS5
	PORT_BIT(0x20, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_A5
	PORT_BIT(0x40, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_AS5
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_B5

	PORT_START("key_row_14")  // C4
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_OTHER) PORT_GM_C6
	PORT_BIT(0xfe, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("key_row_15")
	PORT_BIT(0xff, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("footswitch")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_OTHER) PORT_NAME("CONTROL FOOTSWITCH") PORT_CODE(KEYCODE_SPACE)

	PORT_START("wheel_0")  // Pitch wheel. R1, 100K linear.
	PORT_BIT(0xff, 50, IPT_PADDLE) PORT_NAME("PITCH") PORT_MINMAX(0, 100)
		PORT_SENSITIVITY(30) PORT_KEYDELTA(15) PORT_CENTERDELTA(30)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(sixtrak_state::wheel_moved), WHEEL_PITCH)

	PORT_START("wheel_1")  // Mod wheel. R2, 100K linear.
	PORT_ADJUSTER(0, "MOD")
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(sixtrak_state::wheel_moved), WHEEL_MOD)

	PORT_START("track_volume_knob")  // Knob, R119(?), 10K linear.
	PORT_ADJUSTER(100, "TRACK VOL")

	PORT_START("speed_knob")  // Knob, R115, 10K linear.
	PORT_ADJUSTER(50, "SPEED")

	PORT_START("value_knob")  // Knob, R163(?), 10K linear.
	PORT_ADJUSTER(0, "VALUE")

	PORT_START("tune_knob")  // Knob, R138(?), 10K linear.
	PORT_ADJUSTER(50, "TUNE")

	PORT_START("master_volume_knob")  // Knob, R197, 10K audio taper.
	PORT_ADJUSTER(100, "MASTER VOL")
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(sixtrak_state::master_volume_changed), 0)

	// The default value is for a calibration well within spec.
	// Required offset when inverting 0V: +/- 0.9 mV.
	// Actual offset with the default below: -0.008 mV.
	PORT_START("trimmer_dac_inverter")  // Trimmer, R1205, 100k linear.
	PORT_ADJUSTER(111, "TRIMMER: DAC INVERTER") PORT_MINMAX(0, 255)
		PORT_CHANGED_MEMBER(DEVICE_SELF, FUNC(sixtrak_state::dac_trimmer_changed), 0)
INPUT_PORTS_END

// The firmware version can be displayed by pressing RECORD TRACK and SELECT 5.
ROM_START(sixtrak)
	ROM_DEFAULT_BIOS("v11")
	ROM_SYSTEM_BIOS(0, "v11", "Six-Trak V11 (1985)")  // Last official release.
	ROM_SYSTEM_BIOS(1, "v9", "Six-Trak V9")

	ROM_REGION(0x4000, "maincpu", 0)  // U128 (27128)
	ROMX_LOAD("trak-11.bin", 0x000000, 0x004000, CRC(4d361d4f) SHA1(f14d4291c1a6a3e9462ae9786641cef11a9aac9a), ROM_BIOS(0))
	ROMX_LOAD("trak-9.bin", 0x000000, 0x004000, CRC(d1e6261c) SHA1(bdad57290c24a9ce02c3a5161f8d12f8f96fc74a), ROM_BIOS(1))

	ROM_REGION(0x1000, "nvram_a", ROMREGION_ERASE00)
	ROMX_LOAD("nvram_a-11.bin", 0x000000, 0x001000, CRC(e36e5a14) SHA1(e31fc4fb23ddb34374c785233980023e01a4bffa), ROM_BIOS(0))
	ROMX_LOAD("nvram_a-9.bin", 0x000000, 0x001000, CRC(4dac5eea) SHA1(7afd40b7a79ac0d9e8f081766391232354fb7028), ROM_BIOS(1))

	ROM_REGION(0x800, "nvram_b", ROMREGION_ERASE00)
	ROMX_LOAD("nvram_b-11.bin", 0x000000, 0x000800, CRC(ea2e2fb9) SHA1(d70bf42adf33b2e9970ffd5f1ef3e57217ce349d), ROM_BIOS(0))
	ROMX_LOAD("nvram_b-9.bin", 0x000000, 0x000800, CRC(ea2e2fb9) SHA1(d70bf42adf33b2e9970ffd5f1ef3e57217ce349d), ROM_BIOS(1))
ROM_END

// Firmware versions V11 and V9 are compatible with both Rev A and Rev B / C of
// the Six-Trak.
ROM_START(sixtraka)
	ROM_DEFAULT_BIOS("v11")
	ROM_SYSTEM_BIOS(0, "v11", "Six-Trak V11 (1985)")  // Last official release.
	ROM_SYSTEM_BIOS(1, "v9", "Six-Trak V9")

	ROM_REGION(0x4000, "maincpu", 0)  // U128 (27128)
	ROMX_LOAD("trak-11.bin", 0x000000, 0x004000, CRC(4d361d4f) SHA1(f14d4291c1a6a3e9462ae9786641cef11a9aac9a), ROM_BIOS(0))
	ROMX_LOAD("trak-9.bin", 0x000000, 0x004000, CRC(d1e6261c) SHA1(bdad57290c24a9ce02c3a5161f8d12f8f96fc74a), ROM_BIOS(1))

	ROM_REGION(0x1000, "nvram_a", ROMREGION_ERASE00)
	ROMX_LOAD("nvram_a-11.bin", 0x000000, 0x001000, CRC(e36e5a14) SHA1(e31fc4fb23ddb34374c785233980023e01a4bffa), ROM_BIOS(0))
	ROMX_LOAD("nvram_a-9.bin", 0x000000, 0x001000, CRC(4dac5eea) SHA1(7afd40b7a79ac0d9e8f081766391232354fb7028), ROM_BIOS(1))

	ROM_REGION(0x800, "nvram_b", ROMREGION_ERASE00)
	ROMX_LOAD("nvram_b-11.bin", 0x000000, 0x000800, CRC(ea2e2fb9) SHA1(d70bf42adf33b2e9970ffd5f1ef3e57217ce349d), ROM_BIOS(0))
	ROMX_LOAD("nvram_b-9.bin", 0x000000, 0x000800, CRC(ea2e2fb9) SHA1(d70bf42adf33b2e9970ffd5f1ef3e57217ce349d), ROM_BIOS(1))
ROM_END

}  // anonymous namespace

SYST(1984, sixtrak, 0, 0, sixtrak_rev_b, sixtrak, sixtrak_state, empty_init, "Sequential Circuits", "Six-Trak (Model 610) Rev B/C", MACHINE_IMPERFECT_SOUND | MACHINE_SUPPORTS_SAVE)
SYST(1984, sixtraka, sixtrak, 0, sixtrak_rev_a, sixtrak, sixtrak_state, empty_init, "Sequential Circuits", "Six-Trak (Model 610) Rev A", MACHINE_IMPERFECT_SOUND | MACHINE_SUPPORTS_SAVE)
