/// Provides an implementation of the [IndieAuth](https://indieauth.spec.indieweb.org) standard.
///
/// The implementations in this crate allows for one to
/// obtain tokens, verify them, and perform other common operations.
///
/// # Requesting A Token
/// In order to request a token or profile, one needs to construct an authorization URL
/// that'll allow the user to provide you with an authorization code.
///
/// ## Using Classic Endpoint Discovery
/// The classic discovery method allows you to explicitly provide the authorization and token endpoints:
///
/// ```no_run
/// use indieweb::standards::indieauth::{Client, EndpointDiscovery};
/// use indieweb::http::reqwest::Client as HttpClient;
/// use url::Url;
///
/// # async fn example() -> Result<(), indieweb::Error> {
/// let http_client = HttpClient::default();
/// let _me: Url = "https://jacky.wtf".parse().unwrap();
///
/// let client = Client::<HttpClient>::builder()
///     .id("https://jacky.wtf")
///     .client(http_client)
///     .discovery(EndpointDiscovery::Classic {
///         authorization: "https://jacky.wtf/auth/auth".parse().unwrap(),
///         token: "https://jacky.wtf/auth/token".parse().unwrap(),
///         ticket: None
///     })
///     .build()?;
///
/// // Use the client to build authorization URLs, complete authorization, etc.
/// // See the Client documentation for available methods.
/// # Ok(())
/// # }
/// ```
///
/// # Redeeming an Authorization Code
/// The IndieAuth server will confirm the identity and scopes that the site permits.
/// You'll be redirected back with an authorization code or error information.
///
/// ## Note
/// * The logic for checking CSRF tokens is up to your implementation. It's
///   strongly recommended to do so to prevent forged requests.
///
/// ## Using Metadata Endpoint Discovery
/// The metadata discovery method allows you to provide a metadata endpoint URL that will be
/// automatically queried to discover all required endpoints:
///
/// ```no_run
/// use indieweb::standards::indieauth::{Client, EndpointDiscovery};
/// use indieweb::http::reqwest::Client as HttpClient;
///
/// # async fn example() -> Result<(), indieweb::Error> {
/// let http_client = HttpClient::default();
///
/// let client = Client::<HttpClient>::builder()
///    .id("https://jacky.wtf")
///    .client(http_client)
///    .discovery(EndpointDiscovery::Metadata {
///      metadata: "https://jacky.wtf/.well-known/oauth-authorization-server".parse()?
///    })
///    .build()?;
///
/// // Use client methods to complete authorization, etc.
/// # Ok(())
/// # }
/// ```
///
/// # Verifying A Token
/// Use `introspect_token` to verify a token:
///
/// ```no_run
/// use indieweb::standards::indieauth::{Client, EndpointDiscovery};
/// use indieweb::http::reqwest::Client as HttpClient;
///
/// # async fn example() -> Result<(), indieweb::Error> {
/// let http_client = HttpClient::default();
///
/// let client = Client::<HttpClient>::builder()
///    .id("https://jacky.wtf")
///    .client(http_client)
///    .discovery(EndpointDiscovery::Classic {
///        authorization: "https://jacky.wtf/auth/auth".parse()?,
///        token: "https://jacky.wtf/auth/token".parse()?,
///        ticket: None
///    })
///    .build()?;
///
/// let introspection_endpoint = "https://jacky.wtf/auth/introspect".parse()?;
/// let token = "magic-token";
/// let response = client.introspect_token(&introspection_endpoint, token).await?;
///
/// if response.active {
///     println!("Token is valid");
/// }
/// # Ok(())
/// # }
/// ```
///
/// What's missing from this implementation is logic for things like
/// [AutoAuth](https://indieweb.org/AutoAuth) or
/// [TicketAuth](https://indieweb.org/IndieAuth_Ticket_Auth).
pub mod indieauth;

/// Provides an implementation of the [Micropub](https://micropub.spec.indieweb.org) standard.
///
/// This provides a means of representing a Micropub [query][crate::standards::micropub::Query]
/// and parsing the [responses][crate::standards::micropub::QueryResponse] produced by
/// conforming servers or invoking an [action][micropub::Action] and handling that
/// [response][crate::standards::micropub::ActionResponse] accordingly.
pub mod micropub;

/// Provides an implementation of the [Webmention](https://www.w3.org/TR/webmention/) standard.
///
/// This provides logic for [sending Webmentions][crate::standards::webmention::send],
/// [determining the kind of Webmention][crate::standards::webmention::mention_relationship] and
/// structures around things like [private Webmentions][crate::standards::webmention::PrivateRequest].
///
/// See [send][crate::standards::webmention::send] for more information.
pub mod webmention;

/// Provides an implementation of the [WebSub](https://www.w3.org/TR/websub/) standard.
///
/// This provides functionality for publishers to notify hubs of content updates,
/// subscribers to receive real-time notifications, and hubs to manage subscriptions
/// and distribute content.
#[cfg(feature = "experimental_websub")]
pub mod websub;

/// Provides an implementation of the [Microsub](https://indieweb.org/Microsub-spec) standard.
///
/// This provides functionality for feed consumption and interaction, allowing clients
/// to manage channels, follow feeds, and consume normalized content from various sources.
#[cfg(feature = "experimental_microsub")]
pub mod microsub;

/// Provides an implementation of the [Vouch](https://indieweb.org/Vouch) extension to Webmention.
///
/// This provides anti-spam functionality for Webmention by requiring unknown senders
/// to provide vouch URLs from approved authorities, helping prevent automated spam
/// while maintaining compatibility with legitimate senders.
#[cfg(feature = "experimental_vouch")]
pub mod vouch;
