# How to measure code coverage

This guide shows how to enable the project's built-in coverage targets and how to produce coverage reports manually for a subset of the code using `gcovr`.

## Prerequisites

- Build tools: CMake (>=3.16), Ninja (or other generator), a working C/C++ toolchain.
- Coverage tools: `gcovr` (the CMake coverage helpers check for this).

## Enable coverage builds

1. Create a fresh build directory, configure CMake with the coverage option enabled and build the code:

```bash
cmake -B build-Coverage -DCMAKE_BUILD_TYPE=Debug -DMIR_ENABLE_COVERAGE=ON .
cmake --build build-Coverage
```

2. Build the coverage targets. The repository defines two coverage targets when coverage is enabled:

- `coverage-html` — generates an HTML report (output: `build-Coverage/coverage-html/index.html`).
- `coverage-xml` — generates an XML report (gcovr SonarQube/Cobertura style: `build-Coverage/coverage-xml.xml`).

Build them with CMake or Ninja:

```bash
# build both targets
cmake --build . --target coverage-html -j
cmake --build . --target coverage-xml -j
```

## Measure coverage manually (subset of code)

Sometimes you want to run coverage only for a small portion of the repository or avoid the CMake coverage flow. `gcovr` can generate HTML reports filtered to a subset of sources.

Example — generate an HTML report for WLCS tests over `src/server/frontend_wayland`:

```bash
# Run tests (this produces .gcda/.gcno files)
cd build-Coverage
ctest --output-on-failure --tests-regex wlcs

# Generate an HTML report for a subtree using gcovr (filter is a regex)
gcovr --html-details ../coverage-wlcs.html --root ../ --filter "../src/server/frontend_wayland"
```

## Tips

- If `gcovr` reports no coverage data, verify you built with coverage flags and that tests ran and produced `.gcda`/`.gcno` files.
- Use `--exclude` to remove third-party/packaging directories from reports (e.g. `--exclude '^/usr/'` or `--exclude '^tests/'`).
- Output locations created by the CMake targets are removed by `make clean` (coverage outputs are marked as GENERATED by the helpers).
